#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dollar.h"
#include "actvar.h"
#include "gstring.h"
#include "general.h"
#include "post.h"
#include "xdate.h"
#include "htab.h"

const size_t SOURCE_BLOCK_SIZE = 256;

static unsigned this_year;
static FILE *error_fp = stdout;
static const char *company = NULL;
static xdate current_date;
static dollar *entry_total = NULL;
static dollar *totalling_total = NULL;
static htab *ht = NULL;
static unsigned recursion = 0;

/*---------------------------------------------------------------------------
This function finds the file name in a file specification, puts it into a new
nul-terminated string, and returns a pointer to it.
---------------------------------------------------------------------------*/

char *name_from_filespecs(const char *s)
{
  const char *name = s;
  int c;
  while ((c = *s++) != 0)
  {
    if (c == ':' || c == '\\' || c == '/')
      name = s;
  }
  while (s > name+1 && *s != '.')
    s--;
  gstring g;
  if (*s == '.' && s > name)
    g.append(name, s-name);
  else
    g << name;
  return g.string();
}

/*---------------------------------------------------------------------------
An instance of this class is created for each open journal file.
---------------------------------------------------------------------------*/

class journal
{
  private:
    FILE *fp;
    size_t capacity;
    journal *parent;
  public:
    static journal *current;
    static unsigned number_open;
    char *data;                // nul-terminated line of data
    unsigned long line;        // line number
    bool eof;                  // true if EOF has been read
    char *name;                // name of file
    journal(const char *);
    ~journal();
    bool read(void);           // read a line
};

journal *journal::current = NULL;
unsigned journal::number_open = 0;

/*----------------------------------------------------------------------------
The constructor opens a journal file and puts it on the stack so
journal::current points to it. If the journal file cannot be opened, it is not
put onto the stack and the eof member is set to true.
----------------------------------------------------------------------------*/

journal::journal(const char *specs)
{
  fp = fopen(specs, "r");
  if (fp == NULL)
    eof = true;
  else
  {
    data = new char[(capacity = SOURCE_BLOCK_SIZE)+1];
    line = 0;
    name = name_from_filespecs(specs);
    eof = false;
    parent = current;
    current = this;
    number_open++;
  }
}

/*---------------------------------------------------------------------------
The destructor removes the file from the stack, if it was put onto the stack.
---------------------------------------------------------------------------*/

journal::~journal()
{
  if (fp != NULL)
  {
    fclose(fp);
    delete [] data;
    delete [] name;
    current = parent;
  }
}

/*----------------------------------------------------------------------------
This member function reads a single line of text. It returns false when there
is no more text to be read. The text is put into the buffer beginning at the
"text" member, and the line number is put into "line" member. The text is
terminated by a nul, not by a line feed. All control characters are replaced
by spaces.
----------------------------------------------------------------------------*/

bool journal::read(void)
{
  if (eof)
    return false;
  int c = fgetc(fp);
  if (c == EOF)
  {
    eof = true;
    return false;
  }
  unsigned i = 0;
  while (c != '\n')
  {
    if (i >= capacity)
    {
      char *t = new char[capacity+SOURCE_BLOCK_SIZE+1];
      memcpy(t, data, capacity);
      delete data;
      data = t;
      capacity += SOURCE_BLOCK_SIZE;
    }
    data[i++] = c < ' ' ? ' ' : c;
    c = fgetc(fp);
    if (c == EOF)
    {
      eof = true;
      break;
    }
  };
  data[i] = 0;
  line++;
  return true;
}

/*---------------------------------------------------------------------------
This function scans a string for a name (of an account, variable, journal,
company or directive (but NOT a file), creates a nul-terminated string to hold
the result, and returns a pointer to the first character of the result.

Leading and trailing spaces are removed, and internal runs of spaces are
reduced to single spaces. Capitalization is retained.

If no name is found, the function returns NULL.

The argument s is left pointing to the character following the name.
---------------------------------------------------------------------------*/

char *scan_name(const char *&s)
{
  gstring g;
  s = skip_spaces(s);
  if (!isalpha(*s))
    return NULL;
  while (true)
  {
    do g << *s++; while (isalnum(*s) || *s == '/');
    s = skip_spaces(s);
    if (!isalpha(*s))
      break;
    g << ' ';
  }
  return g.string();
}

dollar *d(const char *s) // diagnostic
{
  bool neg = false;
  if (*s == '-')
  {
    s++;
    neg = true;
  }
  dollar *p = new dollar;
  p->scan(s);
  if (neg)
    p->negate();
  return p;
}

void fatal_error(const char *message)
{
  fprintf(error_fp, "\n%s\n", message);
  exit(1);
}

void numeric_overflow(void)
{
  fatal_error("Numeric overflow");
}

static unsigned number_of_errors = 0;
static unsigned maximum_errors = 0;

void error(const char *message, const char *string = NULL)
{
  if (journal::number_open != 0)
    fprintf(error_fp, "%s %d: ", journal::current->name, journal::current->line);
  for (; *message != 0; message++)
  {
    if (*message == '%' && string != NULL)
      fputs(string, error_fp);
    else
      putc(*message, error_fp);
  }
  putc('\n', error_fp);
  number_of_errors++;
  if (maximum_errors != 0 && number_of_errors >= maximum_errors)
    fatal_error("Too many errors");
}

/*---------------------------------------------------------------------------
Versions of new and delete operators which provide for an orderly shutdown if
memory allocation fails. Also allocation count to check for memory leaks.
---------------------------------------------------------------------------*/

static int allocation_count;

static void insufficient_memory(void)
{
  fatal_error("Insufficient memory");
}

void * operator new (size_t n)
{
  void *p = malloc(n);
  if (p == NULL)
    insufficient_memory();
  allocation_count++;
  return p;
}

void * operator new [] (size_t n)
{
  void *p = malloc(n);
  if (p == NULL)
    insufficient_memory();
  allocation_count++;
  return p;
}

void operator delete (void *p)
{
  if (p != NULL)
  {
    allocation_count--;
    free(p);
  }
}

void operator delete [] (void *p)
{
  if (p != NULL)
  {
    allocation_count--;
    free(p);
  }
}

static void new_account(const char *&s, bool credit)
{
  while (true)
  {
    const char *name = scan_name(s);
    if (name == NULL)
      error("Missing account name");
    else
    {
      actvar *p = new account(name, credit);
      actvar *q = (actvar *) ht->first(p);
      if (q != NULL)
      {
        error("Duplicate account or variable name");
        delete (account *) p;
      }
      else
      {
        ht->insert();
        p->append();
      }
    }
    if (*s == ',')
      s++;
    else
      break;
  }
}

static void new_variable(const char *&s, bool credit)
{
  while (true)
  {
    const char *name = scan_name(s);
    if (name == NULL)
      error("Missing variable name");
    else
    {
      actvar *p = new variable(name, credit);
      actvar *q = (actvar *) ht->first(p);
      if (q != NULL)
      {
        error("Duplicate account or variable name");
        delete (variable *) p;
      }
      else
      {
        ht->insert();
        p->append();
      }
    }
    if (*s == ',')
      s++;
    else
      break;
  }
}

/*---------------------------------------------------------------------------
This function looks up a name in the account and variable table and returns a
pointer to the matching entry, or NULL if there is no match. It deallocates
the name.
---------------------------------------------------------------------------*/

enum lookfor {ACCOUNT, VARIABLE, ACTVAR};

static actvar *lookup(const char *name, lookfor lf)
{
  account *p = new account(name, false);
  actvar *q = (actvar *) ht->first(p);
  if (q != NULL && q->type() == actvar::ALIAS)
    q = ((alias *) q)->referent;
  if (q == NULL || lf == ACCOUNT && q->type() != actvar::ACCOUNT ||
    lf == VARIABLE && q->type() == actvar::ACCOUNT)
  {
    error(
      lf == ACCOUNT ? "Undefined account: %" :
      lf == VARIABLE ? "Undefined variable: %" :
      "Undefined account or variable: %", name);
    q = NULL;
  }
  delete p;
  return q;
}

/*---------------------------------------------------------------------------
This function scans filespecs and opens a file for output, returning a file
pointer (or NULL if the operation failed).
---------------------------------------------------------------------------*/

static FILE *scan_filespecs(const char *&s)
{
  char *mode;
  s = skip_spaces(s);
  if (*s == '+')
  {
    mode = "a";
    s++;
  }
  else
    mode = "w";
  gstring g;
  while (*s != 0 && *s != '*')
    g << *s++;
  const char *specs = g.string();
  FILE *fp = fopen(specs, mode);
  if (fp == NULL)
    error("Can't open file %", specs);
  delete [] specs;
  return fp;
}

static void trial_balance(const char *&s, bool condensed)
{
  FILE *fp = scan_filespecs(s);
  if (fp != NULL)
  {
    unsigned width = (4 * dollar::number_of_digits) / 3;
    actvar *p;
    dollar debit, credit;
    if (company != NULL)
      fprintf(fp, "Company: %s\n\n", company);
    fprintf(fp, "Date: ");
    current_date.print(fp);
    fprintf(fp, "\n\n");
    for (p = actvar::first; p != NULL; p = p->next)
    {
      if (p->type() == actvar::ACCOUNT)
      {
        dollar *amount = p->amount();
        dollar::dcz range = amount->range();
        if (!condensed || range != dollar::ZERO)
        {
          if (range == dollar::CREDIT)
            fputc(' ', fp);
          fprintf(fp, "%s", p->name);
          unsigned i;
          for (i = strlen(p->name); i < 35; i++)
            fputc(' ', fp);
          if (range == dollar::CREDIT)
            putchars(fp, ' ', width+1);
          amount->print(fp, dollar::NO_DEBIT_CREDIT, true, true);
          putc('\n', fp);
        }
        new trial((account *) p, current_date, journal::current->name,
          journal::current->line);
        switch (range)
        {
          case dollar::CREDIT:
            credit += *amount;
            break;
          case dollar::DEBIT:
            debit += *amount;
            break;
        }
        delete amount;
      }
    }
    fprintf(fp, "* ");
    putchars(fp, ' ', 35 - 2);
    putchars(fp, '-', width);
    putchars(fp, ' ', 2);
    putchars(fp, '-', width);
    putc('\n', fp);
    fprintf(fp, "* ");
    putchars(fp, ' ', 35 - 2);
    debit.print(fp, dollar::NO_DEBIT_CREDIT, true, true);
    putchars(fp, ' ', 2);
    credit.print(fp, dollar::NO_DEBIT_CREDIT, true, true);
    putc('\n', fp);
    fprintf(fp, "* ");
    putchars(fp, ' ', 35 - 2);
    putchars(fp, '=', width);
    putchars(fp, ' ', 2);
    putchars(fp, '=', width);
    putc('\n', fp);
    putc('\n', fp);
    fclose(fp);
  }
}

static void ledger(const char *&s, bool condensed)
{
  FILE *fp = scan_filespecs(s);
  if (fp != NULL)
  {
    actvar *p;
    dollar debit, credit;
    if (company != NULL)
      fprintf(fp, "Company: %s\n\n", company);
    fprintf(fp, "Date: ");
    current_date.print(fp);
    fprintf(fp, "\n\n");
    for (p = actvar::first; p != NULL; p = p->next)
    {
      if (p->type() == actvar::ACCOUNT)
      {
        if (!condensed || p->has_postings())
        {
          p->ledger(fp);
        }
      }
    }
    fclose(fp);
  }
}

static void terminate_entry(void)
{
  if (entry_total != NULL)
  {
    if (entry_total->range() != dollar::ZERO)
      error("Unbalanced entry");
    delete entry_total;
    entry_total = NULL;
  }
}

static void not_in_entry(void)
{
  if (entry_total != NULL)
  {
    error("Entry not terminated by blank line or end of journal file");
    delete entry_total;
    entry_total = NULL;
  }
}

static void not_implemented(const char *name)
{
  error("Directive \"%\" not implemented", name);
}

/*---------------------------------------------------------------------------
This function posts the specified amount on the current line to the specified
account. If the amount is NULL, it posts a closing amount.
---------------------------------------------------------------------------*/

static void post_to_account(account *p, dollar *amount)
{
  if (amount == NULL)
  {
    amount = p->amount();
    amount->negate();
  }
  new post((account *) p, current_date, journal::current->name,
    journal::current->line, amount);
  if (entry_total == NULL)
    entry_total = new dollar;
  *entry_total += *amount;
}

/*---------------------------------------------------------------------------
This function posts the specified amount on the current line to the account
with the specified name. If the amount is NULL, it posts a closing amount.
---------------------------------------------------------------------------*/

static void post_to_account(const char *name, dollar *amount)
{
  actvar *q = lookup(name, ACCOUNT);
  if (q == NULL)
    delete amount;
  else
    post_to_account((account *) q, amount);
}

/*---------------------------------------------------------------------------
This function adds the specified amount to the total.
---------------------------------------------------------------------------*/

static void add_to_total(dollar &amount)
{
  if (totalling_total == NULL)
    totalling_total = new dollar;
  *totalling_total += amount;
}

/*---------------------------------------------------------------------------
This function adds a debit or credit amount on the usrrent line into the
running total.
---------------------------------------------------------------------------*/

static void debit_or_credit(const char *&s, bool credit)
{
  dollar amount;
  if (amount.scan(s))
  {
    if (credit)
      amount.negate();
    add_to_total(amount);
  }
  else
    error("Missing or invalid amount");
}

/*---------------------------------------------------------------------------
This function writes a string to the specified output stream, replacing tags
with amounts as required.
---------------------------------------------------------------------------*/

static void write_string(FILE *f, const char *s, bool cents = true)
{
  while (*s!= 0)
  {
    if (*s == '{')
    {
      s++;
      if (*s == '{')
        fputc(*s++, f);
      else if (*s == '-' || *s == ' ' || *s == '=')
      {
        char filler = *s++;
        while (*s == filler)
          s++;
        if (*s != '}')
          error("Unterminated tag");
        else
          s++;
        unsigned width = (4 * dollar::number_of_digits) / 3;
        if (!cents)
          width -= 3;
        fputc(' ', f);
        while (width != 0)
        {
          fputc(filler, f);
          width--;
        }
        fputc(' ', f);
      }
      else
      {
        const char *name = scan_name(s);
        if (name == NULL)
          error("Missing account or variable name");
        else if (*s == ':')
        {
          s++;
          if (*s != '}')
            error("Unterminated tag");
          else
          {
            s++;
            if (match(name, "company"))
              fputs(company, f);
            else if (match(name, "date"))
              current_date.print(f);
          }
          delete [] name;
        }
        else if (*s != '}')
        {
          error("Unterminated tag");
          delete [] name;
        }
        else
        {
          s++;
          actvar *q = lookup(name, ACTVAR);
          if (q != NULL)
          {
            dollar *amount = q->amount();
            amount->print(f, q->credit ? dollar::CREDIT_EXPECTED :
              dollar::DEBIT_EXPECTED, cents, true);
            delete amount;
          }
        }
      }
    }
    else
      fputc(*s++, f);
  }
}

/*---------------------------------------------------------------------------
To take advantage of common code, the add:, subtract: and close: directives
are all handled by the same function.
---------------------------------------------------------------------------*/

const int ADD = 0;
const int SUBTRACT = 1;
const int CLOSE = 2;

static void add_subtract_close(const char *&s, int asc)
{
  const char *name = scan_name(s);
  if (name == NULL)
  {
    error("Missing account or variable name");
    return;
  }
  actvar *first = lookup(name, asc == CLOSE ? ACCOUNT : ACTVAR);
  if (first == NULL)
    return;
  actvar *last = first;
  if (*s == '.')
  {
    do s++; while (*s == '.');
    name = scan_name(s);
    if (name == NULL)
    {
      error("Missing account or variable name");
      return;
    }
    last = lookup(name, first->type() == actvar::ACCOUNT ? ACCOUNT : VARIABLE);
    if (last == NULL)
      return;
  }
  if (first->number > last->number)
  {
    actvar *temp = last;
    last = first;
    first = temp;
  }
  while (true)
  {
    if (asc == CLOSE)
      post_to_account((account *) first, NULL);
    else
    {
      dollar *d = first->amount();
      if (asc == SUBTRACT)
        d->negate();
      add_to_total(*d);
      delete d;
    }
    if (first == last)
      break;
    do
    {
      first = first->next;
    } while (first->type() != last->type());
  }
}

static void post_source_file(const char *specs)
{
  journal src(specs);
  if (src.eof)
  {
    error("Unable to open journal file %", specs);
    return;
  }
  while (journal::current->read())
  {
    bool indented = journal::current->data[0] == ' ' ||
      journal::current->data[0] == '\t';
    // printf("%5d: %s\n", journal::current->line, journal::current->data);
    const char *s = skip_spaces(journal::current->data);
    if (*s == '*')
      continue;
    if (*s == 0)
    {
      terminate_entry();
      continue;
    }
    const char *acct_or_directive = scan_name(s);
    if (acct_or_directive == NULL)
    {
      error("Invalid account or directive");
      continue;
    }
    if (*s == ':')
    {
      s++;
      if (match(acct_or_directive, "add"))
      {
        not_in_entry();
        add_subtract_close(s, ADD);
        ;
      }
      else if (match(acct_or_directive, "alias"))
      {
        not_in_entry();
        const char *alias_name = scan_name(s);
        if (alias_name == NULL)
          error("Missing alias name");
        else if (*s != '=')
        {
          delete [] alias_name;
          error("Missing referent name");
        }
        else
        {
          s++;
          const char *referent_name = scan_name(s);
          if (referent_name == NULL)
          {
            delete [] alias_name;
            error("Missing referent name");
          }
          else
          {
            actvar *p = new account(referent_name, false);
            actvar *q = (actvar *) ht->first(p);
            if (q == NULL)
            {
              error("Undefined variable or account: %", p->name);
              delete p;
            }
            else if (q->type() == actvar::ALIAS)
            {
              error("Double alias: %", alias_name);
              delete p;
            }
            else
            {
              delete p;
              alias *a = new alias(alias_name, q->type() == actvar::ALIAS ?
                ((alias *) q)->referent : q);
              actvar *r = (alias *) ht->first(a);
              if (r != NULL)
              {
                error("Duplicate alias name: %", r->name);
                delete a;
              }
              else
              {
                ht->insert();
                a->append();
              }
            }
          }
        }
      }
      else if (match(acct_or_directive, "begin reverse"))
      {
        not_in_entry();
        not_implemented(acct_or_directive);
        ;
      }
      else if (match(acct_or_directive, "close"))
      {
        add_subtract_close(s, CLOSE);
      }
      else if (match(acct_or_directive, "company"))
      {
        not_in_entry();
        const char *company_name = scan_name(s);
        if (company_name == NULL)
          error("Missing company name");
        else if (company == NULL)
          company = company_name;
        else
        {
          if (!match(company, company_name))
            error("Inconsistent company");
          delete [] company_name;
        }
      }
      else if (match(acct_or_directive, "condensed ledger"))
      {
        not_in_entry();
        ledger(s, true);
      }
      else if (match(acct_or_directive, "condensed trial balance"))
      {
        not_in_entry();
        trial_balance(s, true);
      }
      else if (match(acct_or_directive, "configure"))
      {
        not_in_entry();
        const char *configure = scan_name(s);
        if (configure == NULL)
          error("Missing configuration directive");
        else
        {
          if (match(configure, "comma is decimal"))
          {
            dollar::separator = '.';
            dollar::decimal = ',';
          }
          else if (match(configure, "day/month/year"))
            xdate::order = xdate::DMY;
          else if (match(configure, "digits"))
          {
            unsigned n = 0;
            while (isdigit(*s) && n < 1000)
              n = 10 * n + *s++ - '0';
            if (n < 3 || n > 50)
              error("Invalid number of digits");
            else if (dollar::number_of_digits != 0 &&
              n != dollar::number_of_digits)
            {
              error("Number of digits cannot be changed");
            }
            else
              dollar::number_of_digits = n;
          }
          else if (match(configure, "maximum errors"))
          {
            maximum_errors = 0;
            while (isdigit(*s))
              maximum_errors = 10 * maximum_errors + *s++ - '0';
          }
          else if (match(configure, "month/day/year"))
            xdate::order = xdate::MDY;
          else if (match(configure, "period is decimal"))
          {
            dollar::separator = ',';
            dollar::decimal = '.';
          }
          else
            error("Undefined configuration directive");
          delete [] configure;
        }
      }
      else if (match(acct_or_directive, "credit"))
      {
        not_in_entry();
        debit_or_credit(s, true);
      }
      else if (match(acct_or_directive, "credit account") ||
        match(acct_or_directive, "cr a/c"))
      {
        not_in_entry();
        new_account(s, true);
      }
      else if (match(acct_or_directive, "credit variable"))
      {
        not_in_entry();
        new_variable(s, true);
      }
      else if (match(acct_or_directive, "date"))
      {
        terminate_entry();
        xdate d;
        if (d.scan(s, this_year))
          current_date = d;
        else
          error("invalid date");
      }
      else if (match(acct_or_directive, "debit"))
      {
        not_in_entry();
        debit_or_credit(s, false);
      }
      else if (match(acct_or_directive, "debit account") ||
        match(acct_or_directive, "dr a/c"))
      {
        not_in_entry();
        new_account(s, false);
      }
      else if (match(acct_or_directive, "debit variable"))
      {
        not_in_entry();
        new_variable(s, false);
      }
      else if (match(acct_or_directive, "end reverse"))
      {
        not_implemented(acct_or_directive);
        terminate_entry();
        ;
      }
      else if (match(acct_or_directive, "include"))
      {
        not_in_entry();
        if (recursion > 10)
          error("Include files nested too deep");
        else
        {
          recursion++;
          s = skip_spaces(s);
          gstring g;
          while (*s != 0 && *s != '*')
            g << *s++;
          const char *ispecs = g.string();
          // If ispecs are not a complete file specification, then
          // prepend the directory path of specs
          if (ispecs[0] != '\\' && ispecs[0] != '/' &&
            !(isalpha(ispecs[0]) && ispecs[1] == ':'))
          {
            const char *s = specs + strlen(specs);
            while (s > specs && s[-1] != '\\' && s[-1] != '/' &&
              s[-1] != ':')
            {
              s--;
            }
            gstring h;
            const char *t;
            for (t = specs; t < s; t++)
              h << *t;
            h << ispecs;
            t = ispecs;
            ispecs = h.string();
            delete [] t;
          }
          post_source_file(ispecs);
          delete [] ispecs;
          recursion--;
        }
      }
      else if (match(acct_or_directive, "into"))
      {
        const char *name = scan_name(s);
        if (name == NULL)
          error("Missing account name");
        else if (entry_total == NULL)
        {
          error("No entry before \"into\" directive");
          delete [] name;
        }
        else
        {
          dollar *d = new dollar;
          *d += *entry_total;
          d->negate();
          post_to_account(name, d);
          delete entry_total;
          entry_total = NULL;
        }
      }
      else if (match(acct_or_directive, "journal"))
      {
        not_in_entry();
        char *name = scan_name(s);
        if (name == NULL)
          error("Missing journal");
        else
        {
          delete [] journal::current->name;
          journal::current->name = name;
        }
      }
      else if (match(acct_or_directive, "ledger"))
      {
        not_in_entry();
        ledger(s, false);
      }
      else if (match(acct_or_directive, "message"))
      {
        s = skip_spaces(s);
        // fprintf(error_fp, "%s\n", s);
        write_string(error_fp, s);
        fputc('\n', error_fp);
      }
      else if (match(acct_or_directive, "post credit positive"))
      {
        not_implemented(acct_or_directive);
        not_in_entry();
        ;
      }
      else if (match(acct_or_directive, "post debit positive"))
      {
        not_implemented(acct_or_directive);
        not_in_entry();
        ;
      }
      else if (match(acct_or_directive, "report"))
      {
        not_in_entry();
        s = skip_spaces(s);
        gstring g;
        while (*s != 0 && *s != '*')
        {
          if (*s == ',')
          {
            s++;
            break;
          }
          g << *s++;
        }
        const char *specs = g.string();
        journal src2(specs);
        if (src2.eof)
          error("Unable to open journal file %", specs);
        else
        {
          FILE *fp = scan_filespecs(s);
          if (fp != NULL)
          {
            while (journal::current->read())
            {
              write_string(fp, journal::current->data);
              fputc('\n', fp);
            }
            fclose(fp);
          }
        }
        delete [] specs;
      }
      else if (match(acct_or_directive, "reverse"))
      {
        not_implemented(acct_or_directive);
        not_in_entry();
        ;
      }
      else if (match(acct_or_directive, "round"))
      {
        not_implemented(acct_or_directive);
        not_in_entry();
        ;
      }
      else if (match(acct_or_directive, "subtract"))
      {
        not_in_entry();
        add_subtract_close(s, SUBTRACT);
        ;
      }
      else if (match(acct_or_directive, "total"))
      {
        not_in_entry();
        const char *name = scan_name(s);
        if (name == NULL)
          error("Missing variable name");
        else if (totalling_total == NULL)
        {
          error("No entry before \"total\" directive");
          delete [] name;
        }
        else
        {
          actvar *q = lookup(name, VARIABLE);
          if (q != NULL)
          {
            dollar *temp = totalling_total;
            totalling_total = ((variable *) q)->amt;
            ((variable *) q)->amt = temp;
          }
          delete totalling_total;
          totalling_total = NULL;
        }
      }
      else if (match(acct_or_directive, "trial balance"))
      {
        not_in_entry();
        trial_balance(s, false);
      }
      else if (match(acct_or_directive, "debit variable"))
      {
        not_in_entry();
        new_variable(s, false);
      }
      else
      {
        not_in_entry();
        error("Undefined directive: %", acct_or_directive);
      }
      delete [] acct_or_directive;
    }
    else // account $$$.$$
    {
      dollar *amount = new dollar;
      if (amount->scan(s))
      {
        if (indented)
          amount->negate();
        post_to_account(acct_or_directive, amount);
      }
      else
      {
        delete amount;
        delete [] acct_or_directive;
        error("Invalid amount");
      }
    }
  }
  terminate_entry();

}

int main(int argc, char **argv)
{
  // printf("allocation count = %ld\n", allocation_count);

  current_date.today();
  this_year = current_date.year;

  if (argc < 2)
    fatal_error("No journal file");

  if (argc >= 3)
  {
    FILE *fp = fopen(argv[2], "w");
    if (fp == NULL)
      fatal_error("Can't open error file");
    error_fp = fp;
  }
  else
    error_fp = stdout;

  ht = new htab((htab_match) actvar_match, (htab_code) actvar_code, NULL);

  post_source_file(argv[1]);

  actvar::delete_all();
  delete [] company;
  delete ht;

  // printf("allocation count = %ld\n", allocation_count);

  return number_of_errors != 0;
}

