#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include "xdate.h"
#include "general.h"

enum xdate::order_t xdate::order = xdate::MDY;

/*---------------------------------------------------------------------------
This function edits a date into the form MM/DD/YYYY if the order is
xdate::MDY, or into the form DD/MM/YYYY if the order is DMY.
---------------------------------------------------------------------------*/

void xdate::print(FILE *fp) const
{
  switch (order)
  {
    case MDY:
      fprintf(fp, "%02d/%02d/%04d", month, day, year);
      break;
    case DMY:
      fprintf(fp, "%02d/%02d/%04d", day, month, year);
      break;
  }
}

/*---------------------------------------------------------------------------
This function scans a day and returns its value (or zero if there is an
error). The argument s is left pointing to the character after the day,
and also after a virgule (/), hyphen (-) or comma(,), if one of these
characters immediately follows the day.

Leading spaces are scanned away.
---------------------------------------------------------------------------*/

static unsigned scan_day(const char *&s)
{
  s = skip_spaces(s);
  unsigned n = 0;
  if (isdigit(*s))
  {
    n = *s++ - '0';
    if (isdigit(*s))
      n = 10 * n + *s++ - '0';
    if (*s == '/' || *s == '-' || *s == ',')
        s++;
    if (n > 31)
      n = 0;
  }
  return n;
}

/*---------------------------------------------------------------------------
This function scans a month and returns its value (or zero if there is an
error). The argument s is left pointing to the character after the day,
and also after a virgule (/), hyphen (-) or comma(,), if one of these
characters immediately follows the month.

The month may be represented by a number in the range 1-12, or by the name in
English or Spanish. In the latter case, only the first three letters are
examined, and they are considered without regard to case.

Leading spaces are scanned away.
---------------------------------------------------------------------------*/

static unsigned scan_month(const char *&s)
{
  s = skip_spaces(s);
  unsigned n = 0;
  if (isdigit(*s))
  {
    n = *s++ - '0';
    if (isdigit(*s))
      n = 10 * n + *s++ - '0';
    if (*s == '/' || *s == '-' || *s == ',')
        s++;
    if (n > 12)
      n = 0;
  }
  else if (isalpha(*s))
  {
    char name[4];
    int i = 0;
    while (isalpha(*s))
    {
      if (i < 3)
        name[i++] = toupper(*s);
      s++;
    }
    name[i] = 0;
    static struct
    {
      char name[4];
      unsigned char number;
    } MONTHS[] =
    {
      {"JAN",  1},
      {"ENE",  1},
      {"FEB",  2},
      {"MAR",  3},
      {"APR",  4},
      {"AVR",  4},
      {"MAY",  5},
      {"JUN",  6},
      {"JUL",  7},
      {"AUG",  8},
      {"AGO",  8},
      {"SEP",  9},
      {"SET",  9},
      {"OCT", 10},
      {"NOV", 11},
      {"DEC", 12},
      {"DIC", 12}
    };
    for (i = 0; i < sizeof(MONTHS)/sizeof(MONTHS[0]); i++)
    {
      if (strcmp(MONTHS[i].name, name) == 0)
      {
        n = MONTHS[i].number;
        break;
      }
    }
  }
  return n;
}

/*---------------------------------------------------------------------------
This function scans a date in the order month-day-year or day-month-year,
depending on whether the order is xdate::MDY or xdate::DMY, respectively.
If the year is missing, the default year is used. The argument s is left
pointing to the character following the date. If there is an error, the
function returns the value false.
---------------------------------------------------------------------------*/

bool xdate::scan(const char *&s, unsigned default_year)
{
  switch (order)
  {
    case MDY:
      month = scan_month(s);
      day = scan_day(s);
      break;
    case DMY:
      day = scan_day(s);
      month = scan_month(s);
      break;
  }
  s = skip_spaces(s);
  if (isdigit(*s))
  {
    year = 0;
    while (isdigit(*s))
    {
      year = year * 10 + *s++ - '0';
      if (year > 2400)
        return false;
    }
  }
  else
    year = default_year;
  static unsigned char DAYS_IN_MONTH[13] =
    {0,31,29,31,30,31,30,31,31,30,31,30,31};
  return year > 1600 && month != 0 && day != 0 &&
    day <= DAYS_IN_MONTH[month] && !(month == 2 && day == 29 &&
    (year % 4 != 0 || year % 400 == 0 && year != 2000));
}

/*---------------------------------------------------------------------------
This function gets the current date from the operating system clock.
---------------------------------------------------------------------------*/

void xdate::today(void)
{
  time_t t;
  time(&t);
  struct tm *tt = localtime(&t);
  day = tt->tm_mday;
  month = tt->tm_mon + 1;
  year = tt->tm_year + 1900;
}


